﻿using System;
using System.Runtime.InteropServices;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Interop;
using System.Collections.Generic;
using System.IO;
using System.Diagnostics;
using System.Text;
using Preview;
using System.Windows.Threading;
using Microsoft.Win32;

namespace ScreenWebcams
{
    public partial class MainWindow : Window
    {
        // Registry key and URLs for ZD Soft ScreenWebcams settings and support.
        private const string subKey = "SOFTWARE\\ZD Soft\\ScreenWebcams";

        private DispatcherTimer timerREC = new DispatcherTimer(); // Timer for recording status updates.

        private bool isDragging = false; // Flag to check if dragging is in process.
        private Size sizeView; // Size of the preview area.

        private List<PreviewHost> previews; // List to hold webcam preview hosts.
        private List<Border> overlays; // List to hold overlay elements for webcam previews.

        public MainWindow()
        {
            InitializeComponent();

            // Event handlers for mouse actions on overlay boxes.
            Box0_Overlay.MouseLeftButtonDown += BoxOverlay_MouseLeftButtonDown;

            Box1_Overlay.MouseLeftButtonDown += BoxOverlay_MouseLeftButtonDown;
            Box1_Overlay.MouseMove += BoxOverlay_MouseMove;
            Box1_Overlay.MouseLeftButtonUp += BoxOverlay_MouseLeftButtonUp;
            Box1_Overlay.MouseRightButtonUp += BoxOverlay_MouseRightButtonUp;

            Box2_Overlay.MouseLeftButtonDown += BoxOverlay_MouseLeftButtonDown;
            Box2_Overlay.MouseMove += BoxOverlay_MouseMove;
            Box2_Overlay.MouseLeftButtonUp += BoxOverlay_MouseLeftButtonUp;
            Box2_Overlay.MouseRightButtonUp += BoxOverlay_MouseRightButtonUp;

            Box3_Overlay.MouseLeftButtonDown += BoxOverlay_MouseLeftButtonDown;
            Box3_Overlay.MouseMove += BoxOverlay_MouseMove;
            Box3_Overlay.MouseLeftButtonUp += BoxOverlay_MouseLeftButtonUp;
            Box3_Overlay.MouseRightButtonUp += BoxOverlay_MouseRightButtonUp;
        }

        private void OnLoaded(object sender, RoutedEventArgs e)
        {
            // Initialize preview area size.
            sizeView.Width = Box0.ActualWidth;
            sizeView.Height = Box0.ActualHeight;

            // Initialize the preview and overlay element lists.
            previews = new List<PreviewHost> { Box0, Box1, Box2, Box3 };
            overlays = new List<Border> { Box0_Overlay, Box1_Overlay, Box2_Overlay, Box3_Overlay };

            for (int i = 0; i < previews.Count; i++) { previews[i].block = i; }

            // Make the log file path the same as the process exe file path but different extension.
            string exePath = Process.GetCurrentProcess().MainModule.FileName;
            string logPath = System.IO.Path.ChangeExtension(exePath, ".txt");

            // Create a log file that is helpful for debugging.
            // It can be called before calling ScnLib_Initialize().
            ZDSoft.SDK.ScnLib_SetLogPathW(logPath, true);

            // Initialize the SDK.
            // It must be called before calling most of the other APIs of the SDK.
            // Usually it's called only once at the start of the program.
            ZDSoft.SDK.ScnLib_InitializeW("");

            // Load the last settings stored in the Windows registry.
            ZDSoft.SDK.ScnLib_LoadSettingsW((IntPtr)RegistryHive.CurrentUser, subKey);

            // Switch to the main screen block.
            ZDSoft.SDK.ScnLib_SetLayoutContext(0);

            // Allocate a string to retrieve the video path setting of the main screen block.
            StringBuilder videoPath0 = new StringBuilder(260);

            ZDSoft.SDK.ScnLib_GetVideoPathW(videoPath0, false);

            // Preview the main screen recording that is the only block can be previewed.
            ZDSoft.SDK.ScnLib_PreviewVideo(true, Box0.Handle, false, 0);

            // Switch to the composition screen block.
            ZDSoft.SDK.ScnLib_SetLayoutContext(-1);

            // Allocate a string to retrieve the video path setting of the composition screen block.
            StringBuilder videoPath = new StringBuilder(260);

            ZDSoft.SDK.ScnLib_GetVideoPathW(videoPath, false);

            // Is it in the default state?
            if (videoPath.ToString() == "")
            {
                // In this DEMO, we only want to output a video of the composition screen by default,
                // So we exchange the video path setting of the main screen block and the composition
                // screen block.
                ZDSoft.SDK.ScnLib_SetLayoutContext(0);
                ZDSoft.SDK.ScnLib_SetVideoPathW("");
                ZDSoft.SDK.ScnLib_SetLayoutContext(-1);
                ZDSoft.SDK.ScnLib_SetVideoPathW(videoPath0.ToString());
            }

            // Make the screen capture region frame visible. It's invisible by default.
            // If you don't like the style of the SDK built-in screen capture region frame, you may implement your own.
            ZDSoft.SDK.ScnLib_ShowCaptureRegionFrame(true);

            // Check the position of the 2nd webcam to determine the last position of the webcam overlay group.
            int position = 0, marginX = 0, marginY = 0;

            ZDSoft.SDK.ScnLib_SetLayoutContext(2);
            ZDSoft.SDK.ScnLib_GetWebcamPosition(ref position, ref marginX, ref marginY);
            ZDSoft.SDK.ScnLib_SetLayoutContext(-1);

            if      (position == ZDSoft.SDK.POSITION_LEFT)   MoveBoxGroupToLeft();
            else if (position == ZDSoft.SDK.POSITION_RIGHT)  MoveBoxGroupToRight();
            else if (position == ZDSoft.SDK.POSITION_TOP)    MoveBoxGroupToTop();
            else if (position == ZDSoft.SDK.POSITION_BOTTOM) MoveBoxGroupToBottom();
            else                                             MoveBoxGroupToRight();

            // Prepare a timer for checking the recording status and updating recording time.
            timerREC.Tick += new EventHandler(OnTimerREC);
            timerREC.Interval = TimeSpan.FromSeconds(0.5);

            UpdateUI();
        }

        private void OnClosed(object sender, EventArgs e)
        {
            // Stop the timer.
            timerREC.Stop();

            // Stop recordng if there is one ongoing.
            ZDSoft.SDK.ScnLib_StopRecording();

            // Save current SDK settings into Windows registry.
            ZDSoft.SDK.ScnLib_SaveSettingsW((IntPtr)RegistryHive.CurrentUser, subKey);

            // Uninitialize the SDK.
            // It must be called before exiting the process or there might be resource leak.
            // Usually it's called only once at the end of the program.
            ZDSoft.SDK.ScnLib_Uninitialize();
        }

        private void OnTimerREC(object sender, EventArgs e)
        {
            // Since the SDK doesn't have any callback mechanism, 
            // you need to check the recording status periodically in a timerREC.
            // If it's recording then update the recording time info.
            if (ZDSoft.SDK.ScnLib_IsRecording())
            {
                StringBuilder t = new StringBuilder(11);

                // Get the current recording time string in the format of 'HH:MM:SS'.
                ZDSoft.SDK.ScnLib_GetRecTimeW(t);

                RecTime.Content = t.ToString();
            }
            // If it's not recording then stop recording on failure.
            // The recording may stop automatically if a fatal error occurs.
            // In this case you still need to do some cleanup and UI updates.
            else
            {
                StopRecording(false);

                UpdateUI();
            }
        }

        private void UpdateUI()
        {
            // Updates UI elements based on recording status.
            SaveTo.IsEnabled = !ZDSoft.SDK.ScnLib_IsRecording();

            StringBuilder videoPath = new StringBuilder(260);

            ZDSoft.SDK.ScnLib_GetVideoPathW(videoPath, ZDSoft.SDK.ScnLib_IsRecording());

            OutputFolder.Text = videoPath.ToString();

            // Update the recording control buttons at the bottom.
            if (ZDSoft.SDK.ScnLib_IsRecording())
            {
                StartStop.SetResourceReference(ContentProperty, "StopStr");
            }
            else
            {
                StartStop.SetResourceReference(ContentProperty, "StartStr");
            }

            if (ZDSoft.SDK.ScnLib_IsPaused())
            {
                PauseResume.SetResourceReference(ContentProperty, "ResumeStr");
            }
            else
            {
                PauseResume.SetResourceReference(ContentProperty, "PauseStr");
            }

            PauseResume.IsEnabled = ZDSoft.SDK.ScnLib_IsRecording();
        }

        private void ConfigureBlock(int block)
        {
            IntPtr hwnd = new WindowInteropHelper(Application.Current.MainWindow).Handle;

            // Switch the layout context to the main screen block and store the previous block index.
            int oldBlock = ZDSoft.SDK.ScnLib_SetLayoutContext(block);

            // Configure the settings of the specified block.
            ZDSoft.SDK.ScnLib_ConfigureSettings(hwnd);

            // Restore to the previous layout context.
            ZDSoft.SDK.ScnLib_SetLayoutContext(oldBlock);

            AdjustBoxGroup();
        }

        private string CheckWebcams()
        {
            // Store the previous block index.
            int oldBlock = ZDSoft.SDK.ScnLib_GetLayoutContext();

            // Check whether user has selected the same webcam device for different layout context blocks.
            for (int i = -1; i < 3; i++)
            {
                for (int j = i + 1; j <= 3; j++)
                {
                    int index1 = 0, index2 = 0;

                    ZDSoft.SDK.ScnLib_SetLayoutContext(i);
                    index1 = ZDSoft.SDK.ScnLib_GetSelectedWebcamDevice();

                    ZDSoft.SDK.ScnLib_SetLayoutContext(j);
                    index2 = ZDSoft.SDK.ScnLib_GetSelectedWebcamDevice();

                    if (index1 >= 0 && index2 >= 0 && index1 == index2)
                    {
                        // Restore to the previous layout context.
                        ZDSoft.SDK.ScnLib_SetLayoutContext(oldBlock);

                        return "Block #" + i + " and block #" + j + 
                          " have the same webcam device selected." + 
                          " One of them may not display properly.\n\n" + 
                          " Are you sure you want to proceed?";
                    }
                }
            }

            // Restore to the previous layout context.
            ZDSoft.SDK.ScnLib_SetLayoutContext(oldBlock);

            return "";
        }

        private static void BrowseOutputFolder(string videoPath)
        {
            // Remove the invalid file path chars that are used by SDK-defined variables: <num>, <date> and <time>
            videoPath = videoPath.Replace("<", "");
            videoPath = videoPath.Replace(">", "");

            // Get the folder where your recording videos are saved.
            string videoFolder = System.IO.Path.GetDirectoryName(videoPath.ToString());

            if (Directory.Exists(videoFolder))
            {
                // Browse the video folder in the Windows File Explorer.
                Process.Start("explorer.exe", videoFolder);
            }
        }

        private void StartRecording()
        {
            // Set the video output file of the composition block to the user specified path in the main window.
            ZDSoft.SDK.ScnLib_SetVideoPathW(OutputFolder.Text);

            Mouse.OverrideCursor = Cursors.Wait;

            // Start recording now.
            if (ZDSoft.SDK.ScnLib_StartRecording())
            {
                // Start the timer to check the recording status and update recording time.
                timerREC.Start();
            }
            else
            {
                // Do some cleanup and show an error message if recording failed to start.
                StopRecording(false);

                MessageBox.Show(
                    "Failed to start recording!\n\nPlease check the log file for details.", 
                    "Error", MessageBoxButton.OK, MessageBoxImage.Error);
            }

            Mouse.OverrideCursor = null;

            UpdateUI();
        }

        private void StopRecording(bool bSuccessful)
        {
            // Stop the timer.
            timerREC.Stop();

            Mouse.OverrideCursor = Cursors.Wait;

            // Stop the recording. It's OK to call it even if no recording is in progress.
            ZDSoft.SDK.ScnLib_StopRecording();

            Mouse.OverrideCursor = null;

            ShowResult(bSuccessful);

            UpdateUI();
        }

        private void ShowResult(bool bSuccessful)
        {
            StringBuilder filePath = new StringBuilder(260);

            if (bSuccessful)
            {
                // Get the saved video file path if the recording is done successfully.
                ZDSoft.SDK.ScnLib_GetVideoPathW(filePath, true);
            }
            else
            {
                // Get the saved log file path if the recording is failed.
                ZDSoft.SDK.ScnLib_GetLogPathW(filePath);
            }

            if (filePath.ToString() != "")
            {
                BrowseOutputFolder(filePath.ToString());

                // Play the video file or show the log file.
                if (File.Exists(filePath.ToString()))
                {
                    Process proc = new Process();
                    proc.StartInfo.CreateNoWindow = false;
                    proc.StartInfo.UseShellExecute = true;
                    proc.StartInfo.FileName = filePath.ToString();
                    proc.StartInfo.Verb = "open";
                    proc.Start();
                    proc.Close();
                }
            }
        }

        private int GetBoxIndex(object sender)
        {
            // Determine the overlay index of the sender.
            if (sender is Border boxOverlay)
            {
                for (int i = 0; i < overlays.Count; i++)
                {
                    if (boxOverlay == overlays[i])
                    {
                        return i;
                    }
                }
            }

            return -1;
        }

        private void BoxOverlay_MouseLeftButtonDown(object sender, MouseButtonEventArgs e)
        {
            // Get the overly index of the sender.
            int i = GetBoxIndex(sender);

            if (e.ClickCount == 2) // Double-clicking
            {
                BoxOverlay_MouseLeftButtonUp(sender, e);

                if (i >= 0)
                {
                    // Open the configuration dialog for the double-clicked block.
                    ConfigureBlock(previews[i].block);
                }
            }
            else
            {
                if (i > 0)
                {
                    // Start dragging.
                    isDragging = true;

                    Mouse.OverrideCursor = Cursors.SizeAll;

                    (sender as Border)?.CaptureMouse();
                }
            }
        }

        private void BoxOverlay_MouseMove(object sender, MouseEventArgs e)
        {
            if (isDragging)
            {
                Point currentPosition = e.GetPosition(TopGrid);

                // Move to the nearest edge.
                if (currentPosition.X < (TopGrid.ActualWidth - Box0.ActualWidth) / 2)
                {
                    MoveBoxGroupToLeft();
                }
                else if (currentPosition.X > Box0.ActualWidth + (TopGrid.ActualWidth - Box0.ActualWidth) / 2)
                {
                    MoveBoxGroupToRight();
                }
                else if (currentPosition.Y < (TopGrid.ActualHeight - Box0.ActualHeight) / 2)
                {
                    MoveBoxGroupToTop();
                }
                else if (currentPosition.Y > Box0.ActualHeight + (TopGrid.ActualHeight - Box0.ActualHeight) / 2)
                {
                    MoveBoxGroupToBottom();
                }
            }
        }

        private void BoxOverlay_MouseLeftButtonUp(object sender, MouseButtonEventArgs e)
        {
            if (isDragging)
            {
                isDragging = false;

                Mouse.OverrideCursor = null;

                (sender as Border)?.ReleaseMouseCapture();
            }
        }

        private void BoxOverlay_MouseRightButtonUp(object sender, MouseButtonEventArgs e)
        {
            if (isDragging)
            {
                BoxOverlay_MouseLeftButtonUp(sender, e);
            }
            else
            {
                // Get the overly index of the sender.
                int i = GetBoxIndex(sender);

                if (i > 0)
                {
                    // Remove the right-clicked block.
                    BoxGroup.Children.RemoveAt(i - 1);
                    previews[i].Dispose();
                    previews.RemoveAt(i);
                    overlays.RemoveAt(i);

                    AdjustBoxGroup();
                }
            }
        }

        private void MoveBoxGroupToRight()
        {
            BoxGroup.Orientation = Orientation.Vertical;
            BoxGroup.HorizontalAlignment = HorizontalAlignment.Right;
            BoxGroup.VerticalAlignment = VerticalAlignment.Center;
            AdjustBoxGroup();
        }

        private void MoveBoxGroupToLeft()
        {
            BoxGroup.Orientation = Orientation.Vertical;
            BoxGroup.HorizontalAlignment = HorizontalAlignment.Left;
            BoxGroup.VerticalAlignment = VerticalAlignment.Center;
            AdjustBoxGroup();
        }

        private void MoveBoxGroupToTop()
        {
            BoxGroup.Orientation = Orientation.Horizontal;
            BoxGroup.HorizontalAlignment = HorizontalAlignment.Center;
            BoxGroup.VerticalAlignment = VerticalAlignment.Top;
            AdjustBoxGroup();
        }

        private void MoveBoxGroupToBottom()
        {
            BoxGroup.Orientation = Orientation.Horizontal;
            BoxGroup.HorizontalAlignment = HorizontalAlignment.Center;
            BoxGroup.VerticalAlignment = VerticalAlignment.Bottom;
            AdjustBoxGroup();
        }

        private void AdjustBoxGroup()
        {
            // Switch the layout context to the main screen block and store the previous block index.
            int oldBlock = ZDSoft.SDK.ScnLib_SetLayoutContext(0);

            if (ZDSoft.SDK.ScnLib_IsRecordWebcamOnly())
            {
                // Refresh the video preview if it's in full webcam view mode.
                ZDSoft.SDK.ScnLib_PreviewVideo(true, Box0.Handle, false, 0);
            }

            // Check the count of the remaining webcam boxes.
            int boxCount = BoxGroup.Children.Count;

            if (boxCount > 0)
            {
                // Retrieve the main screen video resolution.
                int videoWidth = 0, videoHeight = 0;

                ZDSoft.SDK.ScnLib_GetVideoResolution(ref videoWidth, ref videoHeight);

                // Determine the proper size of the main box based on the aspect ratio of the main screen.
                double viewWidth = sizeView.Width;
                double viewHeight = sizeView.Height;
                double boxWidth, boxHeight;

                if ((double)videoWidth / (double)videoHeight > sizeView.Width / sizeView.Height)
                {
                    viewHeight = sizeView.Width * videoHeight / videoWidth;
                }
                else
                {
                    viewWidth = sizeView.Height * videoWidth / videoHeight;
                }

                // Calculate the sizes of the remaining webcam boxes.
                if (BoxGroup.Orientation == Orientation.Vertical)
                {
                    boxHeight = viewHeight / boxCount; boxWidth = boxHeight * (1 + boxCount) / 3;
                    videoHeight = videoHeight / boxCount; videoWidth = videoHeight * (1 + boxCount) / 3;
                }
                else
                {
                    boxWidth = viewWidth / boxCount; boxHeight = boxWidth * boxCount / 4;
                    videoWidth = videoWidth / boxCount; videoHeight = videoWidth * boxCount / 4;
                }

                // Adjust the size of the main box.
                Box0.Width = Box0_Overlay.Width = viewWidth;
                Box0.Height = Box0_Overlay.Height = viewHeight;

                // Adjust the sizes of the webcam boxes.
                Box1.Width = Box2.Width = Box3.Width = Box1_Overlay.Width = Box2_Overlay.Width = Box3_Overlay.Width = boxWidth;
                Box1.Height = Box2.Height = Box3.Height = Box1_Overlay.Height = Box2_Overlay.Height = Box3_Overlay.Height = boxHeight;

                // Calculate the position of the remaining webcam views.
                if (BoxGroup.HorizontalAlignment == HorizontalAlignment.Left)
                {
                    BoxGroup.Margin = new Thickness((TopGrid.ActualWidth - viewWidth) / 2 - boxWidth, 0, 0, 0);

                    int[][] leftAlignments =
                    {
                        new int[] {},
                        new int[] { ZDSoft.SDK.POSITION_LEFT },
                        new int[] { ZDSoft.SDK.POSITION_TOP_LEFT, ZDSoft.SDK.POSITION_BOTTOM_LEFT },
                        new int[] { ZDSoft.SDK.POSITION_TOP_LEFT, ZDSoft.SDK.POSITION_LEFT, ZDSoft.SDK.POSITION_BOTTOM_LEFT }
                    };

                    for (int i = 0; i < boxCount; i++)
                    {
                        previews[i + 1].position = leftAlignments[boxCount][i];
                        previews[i + 1].marginX = -videoWidth;
                        previews[i + 1].marginY = 0;
                    }
                }
                else if (BoxGroup.HorizontalAlignment == HorizontalAlignment.Right)
                {
                    BoxGroup.Margin = new Thickness(0, 0, (TopGrid.ActualWidth - viewWidth) / 2 - boxWidth, 0);

                    int[][] rightAlignments =
                    {
                        new int[] {},
                        new int[] { ZDSoft.SDK.POSITION_RIGHT },
                        new int[] { ZDSoft.SDK.POSITION_TOP_RIGHT, ZDSoft.SDK.POSITION_BOTTOM_RIGHT },
                        new int[] { ZDSoft.SDK.POSITION_TOP_RIGHT, ZDSoft.SDK.POSITION_RIGHT, ZDSoft.SDK.POSITION_BOTTOM_RIGHT }
                    };

                    for (int i = 0; i < boxCount; i++)
                    {
                        previews[i + 1].position = rightAlignments[boxCount][i];
                        previews[i + 1].marginX = -videoWidth;
                        previews[i + 1].marginY = 0;
                    }
                }
                else if (BoxGroup.VerticalAlignment == VerticalAlignment.Top)
                {
                    BoxGroup.Margin = new Thickness(0, (TopGrid.ActualHeight - viewHeight) / 2 - boxHeight, 0, 0);

                    int[][] topAlignments =
                    {
                        new int[] {},
                        new int[] { ZDSoft.SDK.POSITION_TOP },
                        new int[] { ZDSoft.SDK.POSITION_TOP_LEFT, ZDSoft.SDK.POSITION_TOP_RIGHT },
                        new int[] { ZDSoft.SDK.POSITION_TOP_LEFT, ZDSoft.SDK.POSITION_TOP, ZDSoft.SDK.POSITION_TOP_RIGHT }
                    };

                    for (int i = 0; i < boxCount; i++)
                    {
                        previews[i + 1].position = topAlignments[boxCount][i];
                        previews[i + 1].marginX = 0;
                        previews[i + 1].marginY = -videoHeight;
                    }
                }
                else if (BoxGroup.VerticalAlignment == VerticalAlignment.Bottom)
                {
                    BoxGroup.Margin = new Thickness(0, 0, 0, (TopGrid.ActualHeight - viewHeight) / 2 - boxHeight);

                    int[][] bottomAlignments =
                    {
                        new int[] {},
                        new int[] { ZDSoft.SDK.POSITION_BOTTOM },
                        new int[] { ZDSoft.SDK.POSITION_BOTTOM_LEFT, ZDSoft.SDK.POSITION_BOTTOM_RIGHT },
                        new int[] { ZDSoft.SDK.POSITION_BOTTOM_LEFT, ZDSoft.SDK.POSITION_BOTTOM, ZDSoft.SDK.POSITION_BOTTOM_RIGHT }
                    };

                    for (int i = 0; i < boxCount; i++)
                    {
                        previews[i + 1].position = bottomAlignments[boxCount][i];
                        previews[i + 1].marginX = 0;
                        previews[i + 1].marginY = -videoHeight;
                    }
                }

                // Preview and position the remaining webcam views.
                for (int i = 1; i < previews.Count; i++)
                {
                    ZDSoft.SDK.ScnLib_SetLayoutContext(previews[i].block);

                    if (ZDSoft.SDK.ScnLib_GetSelectedWebcamDevice() >= 0)
                    {
                        ZDSoft.SDK.ScnLib_PreviewWebcam(true, previews[i].Handle, false, 0);
                    }
                    else
                    {
                        ZDSoft.SDK.ScnLib_PreviewWebcam(false, (IntPtr)0, false, 0);
                    }

                    ZDSoft.SDK.ScnLib_SetWebcamPosition(previews[i].position, previews[i].marginX, previews[i].marginY);
                    ZDSoft.SDK.ScnLib_SetWebcamViewSize(videoWidth, videoHeight);
                }
            }

            // Restore the current layout context to the saved block index.
            ZDSoft.SDK.ScnLib_SetLayoutContext(oldBlock);
        }

        private void OnSelectRegionWindow(object sender, RoutedEventArgs e)
        {
            // Hide the main window before selection.
            this.Hide();

            // Provide the user with a built-in on-screen region selection tool to select
            // a specific window or a rectangular area on the screen.
            int left = 0, top = 0, right = 0, bottom = 0; IntPtr hwnd = (IntPtr)0;

            if (ZDSoft.SDK.ScnLib_SelectCaptureRegionW(ref left, ref top, ref right, ref bottom, ref hwnd, ""))
            {
                ZDSoft.SDK.ScnLib_SetCaptureRegion(left, top, right, bottom); // Define the screen area for recording
                ZDSoft.SDK.ScnLib_SetCaptureWnd(hwnd, true); // Bind to the capture window if the user selected one
            }

            // Show the main window after selection.
            this.Show();

            AdjustBoxGroup();
        }

        private void OnSaveTo(object sender, RoutedEventArgs e)
        {
            Microsoft.Win32.SaveFileDialog dlg = new Microsoft.Win32.SaveFileDialog();

            // Supported video file formats are MP4, FLV and AVI.
            dlg.Filter = "MP4 videos|*.mp4|FLV videos|*.flv|AVI videos|*.avi";
            dlg.DefaultExt = ".mp4";
            dlg.InitialDirectory = Environment.GetFolderPath(Environment.SpecialFolder.MyVideos);
            dlg.Title = "Set Output Video File Path";

            // Popup a save file dialog to let user change the output video file path.
            if (dlg.ShowDialog() == true)
            {
                ZDSoft.SDK.ScnLib_SetVideoPathW(dlg.FileName);

                UpdateUI();
            }
        }

        private void OnBrowse(object sender, RoutedEventArgs e)
        {
            if (OutputFolder.Text != "")
            {
                // Open the output folder in the Windows File Explorer.
                BrowseOutputFolder(OutputFolder.Text);
            }
        }

        private void OnSettings(object sender, RoutedEventArgs e)
        {
            // Configure the composition block settings.
            ConfigureBlock(-1);

            UpdateUI();
        }

        private void OnStartStop(object sender, RoutedEventArgs e)
        {
            if (!ZDSoft.SDK.ScnLib_IsRecording())
            {
                // Check for duplicate webcam device selections.
                string warningMsg = CheckWebcams();

                if (warningMsg != "")
                {
                    MessageBoxResult result = MessageBox.Show(
                        warningMsg, "Warning", 
                        MessageBoxButton.YesNo, 
                        MessageBoxImage.Warning, 
                        MessageBoxResult.No);

                    if (result != MessageBoxResult.Yes)
                    {
                        return;
                    }
                }

                // Make sure the current layout is properly applied.
                AdjustBoxGroup();

                // Start recording if no recording is in progress.
                StartRecording();
            }
            else
            {
                // Stop recording if a recording is in progress.
                StopRecording(true);
            }
        }

        private void OnPauseResume(object sender, RoutedEventArgs e)
        {
            if (!ZDSoft.SDK.ScnLib_IsPaused())
            {
                // Pause recording if the recording is not paused.
                ZDSoft.SDK.ScnLib_PauseRecording();
            }
            else
            {
                // Resume recording if the recording is paused.
                ZDSoft.SDK.ScnLib_ResumeRecording();
            }

            UpdateUI();
        }
    }
}

namespace Preview
{
    // Custom host class for handling webcam preview windows.
    public class PreviewHost : HwndHost
    {
        protected IntPtr hwndBox;
        public int block;
        public int position;
        public int marginX;
        public int marginY;

        protected override HandleRef BuildWindowCore(HandleRef hwndParent)
        {
            // Create a clickable button window as a container of the preview.
            hwndBox = CreateWindowEx(0, "BUTTON", "",
                WS_CHILD | WS_CLIPSIBLINGS | WS_VISIBLE | WS_DISABLED | BS_PUSHBUTTON | BS_FLAT | BS_MULTILINE,
                0, 0, 150, 112, hwndParent.Handle,
                (IntPtr)1, IntPtr.Zero, (IntPtr)0);

            // Display a text on the button when it's not previewing.
            SetWindowText(hwndBox, Text);

            return new HandleRef(this, hwndBox);
        }

        protected override void DestroyWindowCore(HandleRef hwnd)
        {
            DestroyWindow(hwnd.Handle);

            // Switch the layout context to the main screen block and store the previous block index.
            int oldBlock = ZDSoft.SDK.ScnLib_SetLayoutContext(block);

            // Unselect the webcam device of this block.
            ZDSoft.SDK.ScnLib_SelectWebcamDevice(-1);

            // Restore to the previous layout context.
            ZDSoft.SDK.ScnLib_SetLayoutContext(oldBlock);
        }

        protected override void Dispose(bool disposing)
        {
            if (hwndBox != IntPtr.Zero)
            {
                PostMessage(hwndBox, WM_CLOSE, IntPtr.Zero, IntPtr.Zero);
                hwndBox = IntPtr.Zero;
            }

            base.Dispose(disposing);
        }

        // Definitions for window creation and management using Windows API.
        [DllImport("user32.dll", SetLastError = true)]
        private static extern IntPtr CreateWindowEx(int dwExStyle, string lpClassName, string lpWindowName, int dwStyle, int x, int y, int nWidth, int nHeight, IntPtr hWndParent, IntPtr hMenu, IntPtr hInstance, IntPtr lpParam);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern bool DestroyWindow(IntPtr hWnd);

        [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Auto)]
        public static extern bool SetWindowText(IntPtr hWnd, string lpString);

        [DllImport("user32.dll", CharSet = CharSet.Auto)]
        public static extern IntPtr PostMessage(IntPtr hwnd, uint msg, IntPtr wParam, IntPtr lParam);

        private const uint WM_CLOSE = 0x0010;
        private const int WS_CHILD = 0x40000000, WS_CLIPSIBLINGS = 0x04000000, WS_VISIBLE = 0x10000000, WS_DISABLED = 0x08000000;
        private const int BS_PUSHBUTTON = 0x00000000, BS_FLAT = 0x00008000, BS_MULTILINE = 0x00002000;

        public static readonly DependencyProperty TextProperty = DependencyProperty.Register(
            "Text",
            typeof(string),
            typeof(PreviewHost),
            new PropertyMetadata(default(string), OnTextChanged));

        public string Text
        {
            get { return (string)GetValue(TextProperty); }
            set { SetValue(TextProperty, value); }
        }

        private static void OnTextChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            // Reflect the element text change to the window.
            if (d is PreviewHost host) { SetWindowText(host.Handle, e.NewValue as string); }
        }
    }
}
